document.addEventListener('DOMContentLoaded', () => {

    // --- Zaha360 Video Control ---
    const videoOverlay = document.getElementById('video-overlay');
    const soundIcon = document.getElementById('sound-icon');
    const zahaIframe = document.getElementById('zaha-video');

    if (videoOverlay && zahaIframe) {
        videoOverlay.addEventListener('click', () => {
            // Unmute using postMessage (YouTube IFrame API)
            zahaIframe.contentWindow.postMessage('{"event":"command","func":"unMute","args":""}', '*');
            zahaIframe.contentWindow.postMessage('{"event":"command","func":"playVideo","args":""}', '*');

            // UI Feedback
            videoOverlay.classList.add('opacity-0', 'pointer-events-none');
        });
    }

    // --- Mobile Menu (Premium) ---
    const btn = document.getElementById('mobile-menu-btn');
    const closeBtn = document.getElementById('mobile-close-btn');
    const menu = document.getElementById('mobile-menu');
    const body = document.body;

    const toggleMenu = (active) => {
        if (active) {
            menu.classList.add('active');
            btn.classList.add('hamburger-active');
            body.style.overflow = 'hidden';
        } else {
            menu.classList.remove('active');
            btn.classList.remove('hamburger-active');
            body.style.overflow = '';
        }
    };

    if (btn && menu) {
        btn.addEventListener('click', () => toggleMenu(true));
        if (closeBtn) closeBtn.addEventListener('click', () => toggleMenu(false));

        // Close menu on link click
        menu.querySelectorAll('a').forEach(link => {
            link.addEventListener('click', () => toggleMenu(false));
        });
    }

    // --- Quote Form Handler ---
    const quoteForm = document.getElementById('quote-form');
    if (quoteForm) {
        quoteForm.addEventListener('submit', async (e) => {
            e.preventDefault();
            const submitBtn = document.getElementById('submit-btn');
            const originalBtnContent = submitBtn.innerHTML;

            // UI Loading state
            submitBtn.disabled = true;
            submitBtn.innerHTML = '<i class="fa-solid fa-spinner fa-spin mr-2"></i> Procesando...';

            const formData = new FormData(quoteForm);

            try {
                const response = await fetch('api/contact.php', {
                    method: 'POST',
                    body: formData
                });

                const result = await response.json();

                if (result.status === 'success') {
                    Swal.fire({
                        title: '¡Excelente!',
                        text: result.message,
                        icon: 'success',
                        background: '#111',
                        color: '#fff',
                        confirmButtonColor: '#ff6600',
                        customClass: {
                            popup: 'rounded-3xl border border-white/10 shadow-2xl'
                        }
                    });
                    quoteForm.reset();
                } else {
                    throw new Error(result.message);
                }
            } catch (error) {
                Swal.fire({
                    title: 'Error',
                    text: error.message || 'Hubo un problema al procesar su solicitud.',
                    icon: 'error',
                    background: '#111',
                    color: '#fff',
                    confirmButtonColor: '#ff6600'
                });
            } finally {
                submitBtn.disabled = false;
                submitBtn.innerHTML = originalBtnContent;
            }
        });
    }

    // --- Smooth Scroll ---
    document.querySelectorAll('a[href^="#"]').forEach(anchor => {
        anchor.addEventListener('click', function (e) {
            e.preventDefault();
            const href = this.getAttribute('href');
            if (href === "#") return;
            const target = document.querySelector(href);
            if (target) {
                const offset = 80;
                const elementPosition = target.getBoundingClientRect().top;
                const offsetPosition = elementPosition + window.pageYOffset - offset;
                window.scrollTo({ top: offsetPosition, behavior: "smooth" });
            }
        });
    });

    // --- Optimized Flipbook (Progressive Loading) ---
    const initFlipbook = async () => {
        const flipbookEl = document.getElementById('flipbook');
        if (!flipbookEl || typeof St === 'undefined' || typeof pdfjsLib === 'undefined') return;

        try {
            const loadingTask = pdfjsLib.getDocument('portafolio.pdf');
            const pdf = await loadingTask.promise;
            const numPages = pdf.numPages;

            // 1. Create skeleton structure immediately for instant "load"
            const pagesFrag = document.createDocumentFragment();
            const pageNodes = [];

            for (let i = 0; i < numPages; i++) {
                const div = document.createElement('div');
                div.className = 'page';
                div.style.backgroundColor = '#1a1a1a';
                div.style.display = 'flex';
                div.style.justifyContent = 'center';
                div.style.alignItems = 'center';
                div.style.overflow = 'hidden';
                div.innerHTML = '<i class="fa-solid fa-spinner fa-spin text-brand-orange text-2xl"></i>'; // Placeholder

                pagesFrag.appendChild(div);
                pageNodes.push(div);
            }

            flipbookEl.innerHTML = '';
            flipbookEl.appendChild(pagesFrag);

            // 2. Initialize Flipbook Engine immediately
            const pageFlip = new St.PageFlip(flipbookEl, {
                width: 500,
                height: 700,
                size: 'stretch',
                minWidth: 300,
                maxWidth: 1000,
                minHeight: 400,
                maxHeight: 1200,
                maxShadowOpacity: 0.5,
                showCover: true,
                mobileScrollSupport: false
            });

            pageFlip.loadFromHTML(pageNodes);

            // 3. Render pages progressively in background
            const renderPage = async (pageNum) => {
                try {
                    const page = await pdf.getPage(pageNum);
                    // Scale 1.0 is faster and usually sufficient for web view
                    const viewport = page.getViewport({ scale: 1.2 });

                    const canvas = document.createElement('canvas');
                    const context = canvas.getContext('2d');
                    canvas.height = viewport.height;
                    canvas.width = viewport.width;

                    // Fit to container style
                    canvas.style.width = '100%';
                    canvas.style.height = '100%';
                    canvas.style.objectFit = 'contain';

                    await page.render({
                        canvasContext: context,
                        viewport: viewport
                    }).promise;

                    // Replace loader with canvas
                    const pageDiv = pageNodes[pageNum - 1];
                    pageDiv.innerHTML = '';
                    pageDiv.appendChild(canvas);
                } catch (err) {
                    console.error("Error rendering page " + pageNum, err);
                }
            };

            // Render first 3 pages quickly (cover + first spread)
            // Then render the rest
            const renderQueue = async () => {
                for (let i = 1; i <= numPages; i++) {
                    await renderPage(i);
                    // Minimal delay to yield to UI thread
                    await new Promise(r => setTimeout(r, 10));
                }
            };
            renderQueue();

        } catch (error) {
            console.error('Error loading PDF:', error);
            const flipbookEl = document.getElementById('flipbook');
            if (flipbookEl) {
                flipbookEl.innerHTML = `
                    <div class="flex flex-col items-center justify-center h-full p-8 text-center border-2 border-brand-orange/30 rounded-xl bg-brand-surface">
                        <i class="fa-solid fa-file-pdf text-5xl text-brand-orange mb-4"></i>
                        <h3 class="text-xl font-bold text-white mb-2">Portafolio Disponible</h3>
                        <p class="text-gray-400 mb-6 text-sm">Puede descargar nuestro portafolio en PDF.</p>
                        <a href="portafolio.pdf" target="_blank" class="btn-premium px-6 py-2 rounded-lg text-white font-bold uppercase text-sm tracking-wider shadow-lg">
                            Descargar PDF
                        </a>
                    </div>`;
            }
        }
    };
    initFlipbook();

    // --- Counters Animation ---
    const counters = document.querySelectorAll('.counter');
    const speed = 200;
    const animateCounters = () => {
        counters.forEach(counter => {
            const updateCount = () => {
                const target = +counter.getAttribute('data-target');
                const count = +counter.innerText.replace('+', '');
                const inc = target / speed;
                if (count < target) {
                    counter.innerText = Math.ceil(count + inc) + (counter.getAttribute('data-target') > 99 ? '+' : '');
                    setTimeout(updateCount, 20);
                } else {
                    counter.innerText = target + (target > 99 ? '+' : '');
                }
            };
            updateCount();
        });
    };
    let counterPlayed = false;
    const statsSection = document.querySelector('.counter')?.closest('section');
    if (statsSection) {
        const observer = new IntersectionObserver((entries) => {
            if (entries[0].isIntersecting && !counterPlayed) {
                animateCounters();
                counterPlayed = true;
            }
        }, { threshold: 0.5 });
        observer.observe(statsSection);
    }

    // --- FAQ Accordion (FontAwesome Updated) ---
    const accordions = document.querySelectorAll('.accordion-header');
    accordions.forEach(acc => {
        acc.addEventListener('click', function () {
            // Close others (optional, keeps it clean)
            accordions.forEach(otherAcc => {
                if (otherAcc !== acc && otherAcc.nextElementSibling.style.maxHeight) {
                    otherAcc.nextElementSibling.style.maxHeight = null;
                    otherAcc.classList.remove('active');
                    const otherIcon = otherAcc.querySelector('.fa-minus');
                    if (otherIcon) {
                        otherIcon.classList.remove('fa-minus');
                        otherIcon.classList.add('fa-plus');
                    }
                }
            });

            this.classList.toggle('active');
            const panel = this.nextElementSibling;
            const icon = this.querySelector('.fa-plus, .fa-minus');

            if (panel.style.maxHeight) {
                panel.style.maxHeight = null;
                panel.classList.remove('active');
                if (icon) {
                    icon.classList.remove('fa-minus');
                    icon.classList.add('fa-plus');
                }
            } else {
                panel.style.maxHeight = panel.scrollHeight + "px";
                panel.classList.add('active');
                if (icon) {
                    icon.classList.remove('fa-plus');
                    icon.classList.add('fa-minus');
                }
            }
        });
    });
});
